function [channel] = ft_channelselection(desired, datachannel, senstype)

% FT_CHANNELSELECTION makes a selection of EEG and/or MEG channel labels.
% This function translates the user-specified list of channels into channel
% labels as they occur in the data. This channel selection procedure can be
% used throughout FieldTrip.
%
% Use as:
%   channel = ft_channelselection(desired, datachannel)
%
% You can specify a mixture of real channel labels and of special strings,
% or index numbers that will be replaced by the corresponding channel
% labels. Channels that are not present in the raw datafile are
% automatically removed from the channel list.
%
% E.g. the input 'channel' can be:
%  'all'     is replaced by all channels in the datafile
%  'gui'     a graphical user interface will pop up to select the channels
%  'C*'      is replaced by all channels that match the wildcard, e.g. C1, C2, C3, ...
%  '*1'      is replaced by all channels that match the wildcard, e.g. C1, P1, F1, ...
%  'M*1'     is replaced by all channels that match the wildcard, e.g. MEG0111, MEG0131, MEG0131, ...
%  'meg'     is replaced by all MEG channels (works for CTF, 4D, Neuromag and Yokogawa)
%  'megref'  is replaced by all MEG reference channels (works for CTF and 4D)
%  'meggrad' is replaced by all MEG gradiometer channels (works for Yokogawa and Neuromag-306)
%  'megmag'  is replaced by all MEG magnetometer channels (works for Yokogawa and Neuromag-306)
%  'eeg'     is replaced by all recognized EEG channels (this is system dependent)
%  'eeg1020' is replaced by 'Fp1', 'Fpz', 'Fp2', 'F7', 'F3', ...
%  'eog'     is replaced by all recognized EOG channels
%  'ecg'     is replaced by all recognized ECG channels
%  'nirs'    is replaced by all channels recognized as NIRS channels
%  'emg'     is replaced by all channels in the datafile starting with 'EMG'
%  'lfp'     is replaced by all channels in the datafile starting with 'lfp'
%  'mua'     is replaced by all channels in the datafile starting with 'mua'
%  'spike'   is replaced by all channels in the datafile starting with 'spike'
%  10        is replaced by the 10th channel in the datafile
%
% Other channel groups are
%   'EEG1010'    with approximately 90 electrodes
%   'EEG1005'    with approximately 350 electrodes
%   'EEGCHWILLA' for Dorothee Chwilla's electrode caps (used at the DCC)
%   'EEGBHAM'    for the 128 channel EEG system used in Birmingham
%   'EEGREF'     for mastoid and ear electrodes (M1, M2, LM, RM, A1, A2)
%   'MZ'         for MEG zenith
%   'ML'         for MEG left
%   'MR'         for MEG right
%   'MLx', 'MRx' and 'MZx' with x=C,F,O,P,T for left/right central, frontal, occipital, parietal and temporal
%
% You can also exclude channels or channel groups using the following syntax
%   {'all', '-POz', '-Fp1', -EOG'}
%
% See also FT_PREPROCESSING, FT_SENSLABEL, FT_MULTIPLOTER, FT_MULTIPLOTTFR,
% FT_SINGLEPLOTER, FT_SINGLEPLOTTFR

% Note that the order of channels that is returned should correspond with
% the order of the channels in the data.

% Copyright (C) 2003-2014, Robert Oostenveld
%
% This file is part of FieldTrip, see http://www.ru.nl/neuroimaging/fieldtrip
% for the documentation and details.
%
%    FieldTrip is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%    FieldTrip is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with FieldTrip. If not, see <http://www.gnu.org/licenses/>.
%
% $Id: ft_channelselection.m 10410 2015-05-20 19:32:43Z roboos $

% this is to avoid a recursion loop
persistent recursion 

if isempty(recursion)
  recursion = false;
end

if nargin<3
  senstype = ft_senstype(datachannel);
end

if ~iscell(datachannel)
  if ischar(datachannel)
    datachannel = {datachannel};
  else
    error('please specify the data channels as a cell-array');
  end
end

if ~ischar(desired) && ~isnumeric(desired) && ~iscell(desired)
  error('please specify the desired channels as a cell-array or a string');
end

% start with the list of desired channels, this will be pruned/expanded
channel = desired;

if length(datachannel)~=length(unique(datachannel))
  warning('discarding non-unique channel names');
  sel = false(size(datachannel));
  for i=1:length(datachannel)
    sel(i) = sum(strcmp(datachannel, datachannel{i}))==1;
  end
  datachannel = datachannel(sel);
end

if any(size(channel) == 0)
  % there is nothing to do if it is empty
  return
end

if ~iscell(datachannel)
  % ensure that a single input argument like 'all' also works
  datachannel = {datachannel};
end

if isnumeric(channel)
  % remove channels tha fall outside the range
  channel = channel(channel>=1 & channel<=numel(datachannel));
  % change index into channelname
  channel = datachannel(channel);
  return
end

if ~iscell(channel)
  % ensure that a single input argument like 'all' also works
  % the case of a vector with channel indices has already been dealt with
  channel = {channel};
end

% ensure that both inputs are column vectors
channel     = channel(:);
datachannel = datachannel(:);

% remove channels that occur more than once, this sorts the channels alphabetically
[channel, indx] = unique(channel);
% undo the sorting, make the order identical to that of the data channels
[dum, indx] = sort(indx);
channel = channel(indx);

[dataindx, chanindx] = match_str(datachannel, channel);
if length(chanindx)==length(channel)
  % there is a perfect match between the channels and the datachannels, only some reordering is needed
  channel = channel(chanindx);
  % no need to look at channel groups
  return
end

% define the known groups with channel labels
labelall    = datachannel;
label1020   = ft_senslabel('eeg1020'); % use external helper function
label1010   = ft_senslabel('eeg1010'); % use external helper function
label1005   = ft_senslabel('eeg1005'); % use external helper function
labelchwilla = {'Fz', 'Cz', 'Pz', 'F7', 'F8', 'LAT', 'RAT', 'LT', 'RT', 'LTP', 'RTP', 'OL', 'OR', 'FzA', 'Oz', 'F7A', 'F8A', 'F3A', 'F4A', 'F3', 'F4', 'P3', 'P4', 'T5', 'T6', 'P3P', 'P4P'}';
labelbham   = {'P9', 'PPO9h', 'PO7', 'PPO5h', 'PPO3h', 'PO5h', 'POO9h', 'PO9', 'I1', 'OI1h', 'O1', 'POO1', 'PO3h', 'PPO1h', 'PPO2h', 'POz', 'Oz', 'Iz', 'I2', 'OI2h', 'O2', 'POO2', 'PO4h', 'PPO4h', 'PO6h', 'POO10h', 'PO10', 'PO8', 'PPO6h', 'PPO10h', 'P10', 'P8', 'TPP9h', 'TP7', 'TTP7h', 'CP5', 'TPP7h', 'P7', 'P5', 'CPP5h', 'CCP5h', 'CP3', 'P3', 'CPP3h', 'CCP3h', 'CP1', 'P1', 'Pz', 'CPP1h', 'CPz', 'CPP2h', 'P2', 'CPP4h', 'CP2', 'CCP4h', 'CP4', 'P4', 'P6', 'CPP6h', 'CCP6h', 'CP6', 'TPP8h', 'TP8', 'TPP10h', 'T7', 'FTT7h', 'FT7', 'FC5', 'FCC5h', 'C5', 'C3', 'FCC3h', 'FC3', 'FC1', 'C1', 'CCP1h', 'Cz', 'FCC1h', 'FCz', 'FFC1h', 'Fz', 'FFC2h', 'FC2', 'FCC2h', 'CCP2h', 'C2', 'C4', 'FCC4h', 'FC4', 'FC6', 'FCC6h', 'C6', 'TTP8h', 'T8', 'FTT8h', 'FT8', 'FT9', 'FFT9h', 'F7', 'FFT7h', 'FFC5h', 'F5', 'AFF7h', 'AF7', 'AF5h', 'AFF5h', 'F3', 'FFC3h', 'F1', 'AF3h', 'Fp1', 'Fpz', 'Fp2', 'AFz', 'AF4h', 'F2', 'FFC4h', 'F4', 'AFF6h', 'AF6h', 'AF8', 'AFF8h', 'F6', 'FFC6h', 'FFT8h', 'F8', 'FFT10h', 'FT10'};
labelref    = {'M1', 'M2', 'LM', 'RM', 'A1', 'A2'}';
labeleog    = datachannel(strncmp('EOG', datachannel, length('EOG')));               % anything that starts with EOG
labeleog    = [labeleog(:); {'HEOG', 'VEOG', 'VEOG-L', 'VEOG-R', 'hEOG', 'vEOG', 'Eye_Ver', 'Eye_Hor'}'];     % or any of these
labelecg    = datachannel(strncmp('ECG', datachannel, length('ECG')));
labelemg    = datachannel(strncmp('EMG', datachannel, length('EMG')));
labellfp    = datachannel(strncmp('lfp', datachannel, length('lfp')));
labelmua    = datachannel(strncmp('mua', datachannel, length('mua')));
labelspike  = datachannel(strncmp('spike', datachannel, length('spike')));
labelnirs   = datachannel(~cellfun(@isempty, regexp(datachannel, sprintf('%s%s', regexptranslate('wildcard','Rx*-Tx*[*]'), '$'))));

% use regular expressions to deal with the wildcards
labelreg = false(size(datachannel));
findreg  = [];
for i=1:length(channel)
  if length(channel{i}) < 1
      continue;
  end
  
  if strcmp((channel{i}(1)), '-') 
    % skip channels to be excluded
    continue;
  end
  
  rexp = sprintf('%s%s', regexptranslate('wildcard',channel{i}), '$');
  lreg = ~cellfun(@isempty, regexp(datachannel, rexp));
  if any(lreg)
    labelreg = labelreg | lreg;  
    findreg  = [findreg; i];
  end  
end

if ~isempty(findreg)
  findreg  = unique(findreg); % remove multiple occurances due to multiple wildcards
  labelreg = datachannel(labelreg);
end

% initialize all the system-specific variables to empty
labelmeg     = [];
labelmeggrad = [];
labelmegref  = [];
labelmegmag  = [];
labeleeg     = [];

switch senstype

  case {'yokogawa', 'yokogawa160', 'yokogawa160_planar', 'yokogawa64', 'yokogawa64_planar', 'yokogawa440', 'yokogawa440_planar'}
    % Yokogawa axial gradiometers channels start with AG, hardware planar gradiometer 
    % channels start with PG, magnetometers start with M
    megax    = strncmp('AG', datachannel, length('AG'));
    megpl    = strncmp('PG', datachannel, length('PG'));
    megmag   = strncmp('M',  datachannel, length('M' ));
    megind   = logical( megax + megpl + megmag);
    labelmeg      = datachannel(megind);
    labelmegmag   = datachannel(megmag);
    labelmeggrad  = datachannel(megax | megpl);

  case {'ctf64'}
    labelml     = datachannel(~cellfun(@isempty, regexp(datachannel, '^SL')));    % left    MEG channels
    labelmr     = datachannel(~cellfun(@isempty, regexp(datachannel, '^SR')));    % right   MEG channels
    labelmeg    = cat(1, labelml, labelmr);
    labelmegref = [datachannel(strncmp('B'  , datachannel, 1));
      datachannel(strncmp('G'  , datachannel, 1));
      datachannel(strncmp('P'  , datachannel, 1));
      datachannel(strncmp('Q'  , datachannel, 1));
      datachannel(strncmp('R'  , datachannel, length('G'  )))];
    
  case {'ctf', 'ctf275', 'ctf151', 'ctf275_planar', 'ctf151_planar'}
    % all CTF MEG channels start with "M"
    % all CTF reference channels start with B, G, P, Q or R
    % all CTF EEG channels start with "EEG"
    labelmeg    = datachannel(strncmp('M'  , datachannel, length('M'  )));
    labelmegref = [datachannel(strncmp('B'  , datachannel, 1));
      datachannel(strncmp('G'  , datachannel, 1));
      datachannel(strncmp('P'  , datachannel, 1));
      datachannel(strncmp('Q'  , datachannel, 1));
      datachannel(strncmp('R'  , datachannel, length('G'  )))];
    labeleeg  = datachannel(strncmp('EEG', datachannel, length('EEG')));

    % Not sure whether this should be here or outside the switch or
    % whether these specifications should be supported for systems
    % other than CTF.
    labelmz   = datachannel(strncmp('MZ' , datachannel, length('MZ' )));    % central MEG channels
    labelml   = datachannel(strncmp('ML' , datachannel, length('ML' )));    % left    MEG channels
    labelmr   = datachannel(strncmp('MR' , datachannel, length('MR' )));    % right   MEG channels
    labelmlc  = datachannel(strncmp('MLC', datachannel, length('MLC')));
    labelmlf  = datachannel(strncmp('MLF', datachannel, length('MLF')));
    labelmlo  = datachannel(strncmp('MLO', datachannel, length('MLO')));
    labelmlp  = datachannel(strncmp('MLP', datachannel, length('MLP')));
    labelmlt  = datachannel(strncmp('MLT', datachannel, length('MLT')));
    labelmrc  = datachannel(strncmp('MRC', datachannel, length('MRC')));
    labelmrf  = datachannel(strncmp('MRF', datachannel, length('MRF')));
    labelmro  = datachannel(strncmp('MRO', datachannel, length('MRO')));
    labelmrp  = datachannel(strncmp('MRP', datachannel, length('MRP')));
    labelmrt  = datachannel(strncmp('MRT', datachannel, length('MRT')));
    labelmzc  = datachannel(strncmp('MZC', datachannel, length('MZC')));
    labelmzf  = datachannel(strncmp('MZF', datachannel, length('MZF')));
    labelmzo  = datachannel(strncmp('MZO', datachannel, length('MZO')));
    labelmzp  = datachannel(strncmp('MZP', datachannel, length('MZP')));

  case {'bti', 'bti248', 'bti248grad', 'bti148', 'bti248_planar', 'bti148_planar'}
    % all 4D-BTi MEG channels start with "A"
    % all 4D-BTi reference channels start with M or G
 
    labelmeg     = datachannel(myregexp('^A[0-9]+$', datachannel));
    labelmegref  = [datachannel(myregexp('^M[CLR][xyz][aA]*$', datachannel)); datachannel(myregexp('^G[xyz][xyz]A$', datachannel)); datachannel(myregexp('^M[xyz][aA]*$', datachannel))];
    labelmegrefa = datachannel(~cellfun(@isempty,strfind(datachannel, 'a')));
    labelmegrefc = datachannel(strncmp('MC', datachannel, 2));
    labelmegrefg = datachannel(myregexp('^G[xyz][xyz]A$', datachannel));
    labelmegrefl = datachannel(strncmp('ML', datachannel, 2));
    labelmegrefr = datachannel(strncmp('MR', datachannel, 2));
    labelmegrefm = datachannel(myregexp('^M[xyz][aA]*$', datachannel));

  case {'neuromag122' 'neuromag122alt'}
    % all neuromag MEG channels start with MEG
    % all neuromag EEG channels start with EEG
    labelmeg = datachannel(strncmp('MEG', datachannel, length('MEG')));
    labeleeg = datachannel(strncmp('EEG', datachannel, length('EEG')));
    
  case {'neuromag306' 'neuromag306alt'}
    % all neuromag MEG channels start with MEG
    % all neuromag EEG channels start with EEG
    % all neuromag-306 gradiometers follow pattern MEG*2,MEG*3
    % all neuromag-306 magnetometers follow pattern MEG*1
    labelmeg = datachannel(strncmp('MEG', datachannel, length('MEG')));
    labeleeg = datachannel(strncmp('EEG', datachannel, length('EEG')));
    
    labelmeggrad = labelmeg(~cellfun(@isempty, regexp(labelmeg, '^MEG.*[23]$')));
    labelmegmag  = labelmeg(~cellfun(@isempty, regexp(labelmeg, '^MEG.*1$')));

  case {'ant128', 'biosemi64', 'biosemi128', 'biosemi256', 'egi32', 'egi64', 'egi128', 'egi256', 'eeg1020', 'eeg1010', 'eeg1005', 'ext1020'}
    % use an external helper function to define the list with EEG channel names
    labeleeg = ft_senslabel(ft_senstype(datachannel));
  
  case {'itab153'}
    % all itab MEG channels start with MAG
    labelmeg = datachannel(strncmp('MAG', datachannel, length('MAG')));

end % switch ft_senstype

% figure out if there are bad channels or channel groups that should be excluded
findbadchannel = strncmp('-', channel, length('-'));      % bad channels start with '-'
badchannel = channel(findbadchannel);
if ~isempty(badchannel)
  for i=1:length(badchannel)
    badchannel{i} = badchannel{i}(2:end);                 % remove the '-' from the channel label
  end
  badchannel = ft_channelselection(badchannel, datachannel); % support exclusion of channel groups
end

% determine if any of the known groups is mentioned in the channel list
findall        = find(strcmp(channel, 'all'));
% findreg (for the wildcards) is dealt with in the channel group specification above
findmeg        = find(strcmpi(channel, 'MEG'));
findemg        = find(strcmpi(channel, 'EMG'));
findecg        = find(strcmpi(channel, 'ECG'));
findeeg        = find(strcmpi(channel, 'EEG'));
findeeg1020    = find(strcmpi(channel, 'EEG1020'));
findeeg1010    = find(strcmpi(channel, 'EEG1010'));
findeeg1005    = find(strcmpi(channel, 'EEG1005'));
findeegchwilla = find(strcmpi(channel, 'EEGCHWILLA'));
findeegbham    = find(strcmpi(channel, 'EEGBHAM'));
findeegref     = find(strcmpi(channel, 'EEGREF'));
findmegref     = find(strcmpi(channel, 'MEGREF'));
findmeggrad    = find(strcmpi(channel, 'MEGGRAD'));
findmegmag     = find(strcmpi(channel, 'MEGMAG'));
findmegrefa    = find(strcmpi(channel, 'MEGREFA'));
findmegrefc    = find(strcmpi(channel, 'MEGREFC'));
findmegrefg    = find(strcmpi(channel, 'MEGREFG'));
findmegrefl    = find(strcmpi(channel, 'MEGREFL'));
findmegrefr    = find(strcmpi(channel, 'MEGREFR'));
findmegrefm    = find(strcmpi(channel, 'MEGREFM'));
findeog        = find(strcmpi(channel, 'EOG'));
findmz         = find(strcmp(channel, 'MZ' ));
findml         = find(strcmp(channel, 'ML' ));
findmr         = find(strcmp(channel, 'MR' ));
findmlc        = find(strcmp(channel, 'MLC'));
findmlf        = find(strcmp(channel, 'MLF'));
findmlo        = find(strcmp(channel, 'MLO'));
findmlp        = find(strcmp(channel, 'MLP'));
findmlt        = find(strcmp(channel, 'MLT'));
findmrc        = find(strcmp(channel, 'MRC'));
findmrf        = find(strcmp(channel, 'MRF'));
findmro        = find(strcmp(channel, 'MRO'));
findmrp        = find(strcmp(channel, 'MRP'));
findmrt        = find(strcmp(channel, 'MRT'));
findmzc        = find(strcmp(channel, 'MZC'));
findmzf        = find(strcmp(channel, 'MZF'));
findmzo        = find(strcmp(channel, 'MZO'));
findmzp        = find(strcmp(channel, 'MZP'));
findnirs       = find(strcmpi(channel, 'NIRS'));
findlfp        = find(strcmpi(channel, 'lfp'));
findmua        = find(strcmpi(channel, 'mua'));
findspike      = find(strcmpi(channel, 'spike'));
findgui        = find(strcmpi(channel, 'gui'));

% remove any occurences of groups in the channel list
channel([
  findall
  findreg
  findmeg
  findemg
  findecg
  findeeg
  findeeg1020
  findeeg1010
  findeeg1005
  findeegchwilla
  findeegbham
  findeegref
  findmegref
  findmeggrad
  findmegmag
  findeog
  findmz
  findml
  findmr
  findmlc
  findmlf
  findmlo
  findmlp
  findmlt
  findmrc
  findmrf
  findmro
  findmrp
  findmrt
  findmzc
  findmzf
  findmzo
  findmzp
  findlfp
  findmua
  findspike
  findnirs
  findgui
  ]) = [];

% add the full channel labels to the channel list
if findall,        channel = [channel; labelall]; end
if findreg,        channel = [channel; labelreg]; end
if findmeg,        channel = [channel; labelmeg]; end
if findecg,        channel = [channel; labelecg]; end
if findemg,        channel = [channel; labelemg]; end
if findeeg,        channel = [channel; labeleeg]; end
if findeeg1020,    channel = [channel; label1020]; end
if findeeg1010,    channel = [channel; label1010]; end
if findeeg1005,    channel = [channel; label1005]; end
if findeegchwilla, channel = [channel; labelchwilla]; end
if findeegbham,    channel = [channel; labelbham]; end
if findeegref,     channel = [channel; labelref]; end
if findmegref,     channel = [channel; labelmegref]; end
if findmeggrad,    channel = [channel; labelmeggrad]; end
if findmegmag,     channel = [channel; labelmegmag]; end
if findmegrefa,    channel = [channel; labelmegrefa]; end
if findmegrefc,    channel = [channel; labelmegrefc]; end
if findmegrefg,    channel = [channel; labelmegrefg]; end
if findmegrefl,    channel = [channel; labelmegrefl]; end
if findmegrefr,    channel = [channel; labelmegrefr]; end
if findmegrefm,    channel = [channel; labelmegrefm]; end
if findeog,        channel = [channel; labeleog]; end
if findmz ,        channel = [channel; labelmz ]; end
if findml ,        channel = [channel; labelml ]; end
if findmr ,        channel = [channel; labelmr ]; end
if findmlc,        channel = [channel; labelmlc]; end
if findmlf,        channel = [channel; labelmlf]; end
if findmlo,        channel = [channel; labelmlo]; end
if findmlp,        channel = [channel; labelmlp]; end
if findmlt,        channel = [channel; labelmlt]; end
if findmrc,        channel = [channel; labelmrc]; end
if findmrf,        channel = [channel; labelmrf]; end
if findmro,        channel = [channel; labelmro]; end
if findmrp,        channel = [channel; labelmrp]; end
if findmrt,        channel = [channel; labelmrt]; end
if findmzc,        channel = [channel; labelmzc]; end
if findmzf,        channel = [channel; labelmzf]; end
if findmzo,        channel = [channel; labelmzo]; end
if findmzp,        channel = [channel; labelmzp]; end
if findlfp,        channel = [channel; labellfp]; end
if findmua,        channel = [channel; labelmua]; end
if findspike,      channel = [channel; labelspike]; end
if findnirs,       channel = [channel; labelnirs]; end

% remove channel labels that have been excluded by the user
badindx = match_str(channel, badchannel);
channel(badindx) = [];

% remove channel labels that are not present in the data
chanindx = match_str(channel, datachannel);

channel  = channel(chanindx);

if findgui
  indx = select_channel_list(datachannel, match_str(datachannel, channel), 'Select channels');
  channel = datachannel(indx);
end

% remove channels that occur more than once, this sorts the channels alphabetically
channel = unique(channel);

if isempty(channel) && ~recursion
  % try whether only lowercase channel labels makes a difference
  recursion = true;
  channel   = ft_channelselection(desired, lower(datachannel));
  recursion = false;
  % undo the conversion to lowercase, this sorts the channels alphabetically
  [c, ia, ib] = intersect(channel, lower(datachannel));
  channel = datachannel(ib);
end

if isempty(channel) && ~recursion
  % try whether only uppercase channel labels makes a difference
  recursion = true;
  channel = ft_channelselection(desired, upper(datachannel));
  recursion = false;
  % undo the conversion to uppercase, this sorts the channels alphabetically
  [c, ia, ib] = intersect(channel, lower(datachannel));
  channel = datachannel(ib);
end

% undo the sorting, make the order identical to that of the data channels
[tmp, indx] = match_str(datachannel, channel);
channel = channel(indx);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% helper function
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function match = myregexp(pat, list)
match = false(size(list));
for i=1:numel(list)
  match(i) = ~isempty(regexp(list{i}, pat, 'once'));
end

